/**
 * \file: sdc.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Secure Data Container
 * \brief : Implementation of non architecture specific functions
 *
 * \author: Ian Molton (ian.molton@codethink.co.uk)
 *      Norbert Uetrecht (nuetrecht@de.adit-jv.com)
 *      Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <string.h>
#include <sdc.h>
#include <private/sdc_arch.h>
#include <private/sdc_intern.h>
#include <sdc/arch/errors.h>

/* Error strings */
static const char *error_strings[] = {
    [SDC_OK] = "SDC_OK - no error",
    [SDC_NO_MEM] = "SDC_NO_MEM - no memory available",
    [SDC_INVALID_VERSION] = "SDC_INVALID_VERSION - Mismatch of kernel and user-space version detected",
    [SDC_CONFIG_MISSING] = "SDC_CONFIG_MISSING - no such config entry",
    [SDC_INVALID_PARAMETER] = "SDC_INVALID_PARAMETER - unspecifided parameters is invalid",
    [SDC_KEY_INVALID] = "SDC_KEY_INVALID - the given key is invalid",
    [SDC_ALG_MODE_INVALID] = "SDC_ALG_MODE_INVALID - the given algorithm or mode is invalid",
    [SDC_IV_INVALID] = "SDC_IV_INVALID - the given IV is invalid",
    [SDC_IN_DATA_INVALID] = "SDC_IN_DATA_INVALID - the given in data is invalid",
    [SDC_OUT_DATA_INVALID] = "SDC_OUT_DATA_INVALID - the given out data is invalid",
    [SDC_AAD_DATA_INVALID] = "SDC_AAD_DATA_INVALID - the given aad data is invalid",
    [SDC_SIG_DATA_INVALID] = "SDC_SIG_DATA_INVALID - the given signature data is invalid",
    [SDC_SESSION_INVALID] = "SDC_SESSION_INVALID - the given session is invalid",
    [SDC_PERM_INVALID] = "SDC_PERM_INVALID - the given permissions are invalid",
    [SDC_FORMATTED_DATA_INVALID] = "SDC_FORMATTED_DATA_INVALID - an issue with the given formatted data has been detected",
    [SDC_KEYLEN_INVALID] = "SDC_KEYLEN_INVALID - the given key-length is invalid",
    [SDC_DAEMON_COMMUNICATION_ERROR] = "SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed",
    [SDC_DAEMON_REQUEST_REJECTED] = "SDC_DAEMON_REQUEST_REJECTED - the request has been rejected by the daemon",
    [SDC_DAEMON_INVALID_RESPONSE] = "SDC_DAEMON_INVALID_RESPONSE - checking response of daemon failed",
    [SDC_DAEMON_TIMEOUT_ERROR] = "SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time",
    [SDC_DAEMON_SOCKET_CREATION_FAILED] = "SDC_DAEMON_SOCKET_CREATION_FAILED - Daemon failed to create socket",
    [SDC_KEY_STORAGE_ALREADY_PRESENT] = "SDC_KEY_STORAGE_ALREADY_PRESENT - Key-storage does already exist",
    [SDC_KEY_STORAGE_ACCESS_FAILED] = "SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access the key storage",
    [SDC_KEY_STORAGE_CORRUPT] = "SDC_KEY_STORAGE_CORRUPT - daemon failed to load corrupted key storage",
    [SDC_KEY_CONTAINER_IMPORT_FAILED] = "SDC_KEY_CONTAINER_IMPORT_FAILED - daemon failed to import a key container",
    [SDC_KID_NOT_AVAILABLE] = "SDC_KID_NOT_AVAILABLE - the KID does not exist",
    [SDC_KID_EXISTS] = "SDC_KID_EXISTS - could not install key as the KID is already used",
    [SDC_KEY_UNSET] = "SDC_KEY_UNSET - operation fails as no key has been assigned to a session",
    [SDC_KEY_LOCKED] = "SDC_KEY_LOCKED - operation aborted as key or key storage is currently locked",
    [SDC_KEY_NOT_READY] = "SDC_KEY_NOT_READY - operation aborted as key is not ready to be used",
    [SDC_KEY_TAMPERED] = "SDC_KEY_TAMPERED - operation aborted as the key seems to be tampered",
    [SDC_ACCESS_DENIED] = "SDC_ACCESS_DENIED - the process has not sufficient permissions to perform the requested operation",
    [SDC_OP_NOT_SUPPORTED] = "SDC_OP_NOT_SUPPORTED - Operation not supported",
    [SDC_AUTHENTICATION_FAILED] = "SDC_AUTHENTICATION_FAILED - authentication of data failed",
    [SDC_PADDING_ERROR] = "SDC_PADDING_ERROR - invalid padding detected when removing padding",
    [SDC_INTERNAL_ERROR] = "SDC_INTERNAL_ERROR - these errors should not occur if all API layers are implemented correctly",
    [SDC_NOT_SUPPORTED] = "SDC_NOT_SUPPORTED - the function called is currently not supported for the current architecture",
    [SDC_UNKNOWN_ERROR] = "SDC_UNKNOWN_ERROR - ?",
    [SDC_CONFIG_ACCESS_FAILED] = "SDC_CONFIG_ACCESS_FAILED - failed to read sdc config",
    [SDC_CONFIG_INVALID] = "SDC_CONFIG_INVALID - entry in SDC config is invalid",
    [SDC_DAEMON_MKDIR_FAILED] = "SDC_DAEMON_MKDIR_FAILED - SDC daemon failed to create missing directory",
    [SDC_KEY_INFO_INVALID] = "SDC_KEY_INFO_INVALID - failed to fill key info",
    [SDC_AUTOLOAD_KEY_UNSUPPORTED] = "SDC_AUTOLOAD_KEY_UNSUPPORTED - The key type of the formatted data is not supported",
    [SDC_AUTOLOAD_KEY_WITH_SECRET] = "SDC_AUTOLOAD_KEY_WITH_SECRET - The key of the formatted data requires a secret modifier to be specified",
    [SDC_MODIFIER_INVALID] = "SDC_MODIFIER_INVALID - the modifier is invalid for the key or its length is invalid",
    [SDC_DGST_DATA_INVALID] = "SDC_DGST_DATA_INVALID - invalid digest",
    [SDC_KEY_FMT_INVALID] = "SDC_KEY_FMT_INVALID - the given key-format is invalid",
    [SDC_KEY_ENC_INVALID] = "SDC_KEY_ENC_INVALID - the encoding of the key secret is invalid",
    [SDC_OP_SEQUENCE_INVALID] = "SDC_OP_SEQUENCE_INVALID - the operation sequence is invalid",

#ifdef ARCH_ERROR_STRINGS
    ARCH_ERROR_STRINGS
#endif
};

/* constants */
static const char *sdc_key_fmt_names[] = {
    [SDC_KEY_FMT_SIMPLE_SYM]  = "simple symmetric key",
    [SDC_KEY_FMT_RSA_PUBLIC]  = "RSA public key",
    [SDC_KEY_FMT_RSA_PRIVATE] = "RSA private key"
};

/* struct used for the definition of key length */
struct key_len_bits_bytes_entry {
    size_t bits;
    size_t bytes;
};

static const struct key_len_bits_bytes_entry key_len_bits_bytes[] = {
    { 0, 0 },           /* SDC_KEY_LEN_UNKNOWN */
    { 56, 7 },          /* SDC_KEY_LEN_56bit */
    { 64, 8 },          /* SDC_KEY_LEN_64bit */
    { 80, 10 },         /* SDC_KEY_LEN_80bit */
    { 96, 12 },         /* SDC_KEY_LEN_96bit */
    { 112, 14 },        /* SDC_KEY_LEN_112bit */
    { 128, 16 },        /* SDC_KEY_LEN_112bit */
    { 192, 24 },        /* SDC_KEY_LEN_192bit */
    { 256, 32 },        /* SDC_KEY_LEN_256bit */
    { 384, 48 },        /* SDC_KEY_LEN_384bit */
    { 512, 64 },        /* SDC_KEY_LEN_512bit */
    { 1024, 128 },      /* SDC_KEY_LEN_1024bit */
    { 2048, 256 },      /* SDC_KEY_LEN_2048bit */
    { 4096, 512 },      /* SDC_KEY_LEN_4096bit */
};

#define CHECK_KEY_LEN_BITS_BYTES_ARR_COMPLETE (STATIC_ASSERT_EXPR(ARRAY_SIZE(key_len_bits_bytes) == SDC_KEY_LEN_END))

/* Functions */

sdc_error_t sdc_overwrite_secret(uint8_t *secret, size_t secretlen)
{
    if (secret == NULL)
        return SDC_INVALID_PARAMETER;
    if (secretlen == 0)
        return SDC_INVALID_PARAMETER;

    sdc_intern_overwrite_secret(secret, secretlen);

    return SDC_OK;
}

/* External available interface as defined in sdc.h */
sdc_error_t sdc_plain_buffer_free (uint8_t *plain_data, const size_t plain_len)
{
    if (plain_data == NULL)
        return SDC_INVALID_PARAMETER;
    if (plain_len == 0)
        return SDC_INVALID_PARAMETER;

    sdc_intern_overwrite_secret(plain_data, plain_len);

    free(plain_data);

    return SDC_OK;
}

sdc_error_t sdc_cipher_buffer_free (uint8_t *cipher_data, const size_t cipher_len)
{
    if (cipher_data == NULL)
        return SDC_INVALID_PARAMETER;
    if (cipher_len == 0)
        return SDC_INVALID_PARAMETER;

    free(cipher_data);

    return SDC_OK;
}

/* External available interface as defined in sdc.h */
const char* sdc_get_error_string (sdc_error_t err)
{
    const char *c = NULL;

    if (err < ARRAY_SIZE(error_strings))
        c = error_strings[err];

    if (c == NULL) {
        if (err >= SDC_ARCHITECTURE_SPECIFIC_ERROR_01)
            c = "Some architecture specific error";
        else
            c = "No error string available";
    }
    return c;
}


static sdc_error_t check_sdc_key_len_t(sdc_key_len_t keylen)
{
    if ((keylen >= SDC_KEY_LEN_FIRST) && (keylen < SDC_KEY_LEN_END))
        return SDC_OK;

    return SDC_KEY_FMT_INVALID;
}

static sdc_error_t sdc_key_len_get_bits_bytes(sdc_key_len_t keylen_val,
                                              size_t *keylen_bits,
                                              size_t *keylen_bytes)
{
    sdc_error_t err;
    const struct key_len_bits_bytes_entry *entry;

    err = SDC_OK;
    *keylen_bits = 0;
    *keylen_bytes = 0;

    /* check size of array */
    CHECK_KEY_LEN_BITS_BYTES_ARR_COMPLETE;

    if ((keylen_val < SDC_KEY_LEN_FIRST) &&
        (keylen_val >= SDC_KEY_LEN_END)) {
        err = SDC_KEYLEN_INVALID;
    } else {
        entry = &key_len_bits_bytes[keylen_val];

        *keylen_bits  = entry->bits;
        *keylen_bytes = entry->bytes;
    }
    return err;
}

sdc_error_t sdc_key_len_get_bits(sdc_key_len_t keylen_val, size_t *keylen_bits)
{
    size_t bits;
    size_t bytes;
    sdc_error_t err;

    if (!keylen_bits)
        return SDC_INVALID_PARAMETER;

    err = sdc_key_len_get_bits_bytes(keylen_val, &bits, &bytes);

    if (err == SDC_OK) {
        *keylen_bits = bits;
    }

    return err;
}

sdc_error_t sdc_key_len_get_bytes(sdc_key_len_t keylen_val, size_t *keylen_bytes)
{
    size_t bits;
    size_t bytes;
    sdc_error_t err;

    if (!keylen_bytes)
        return SDC_INVALID_PARAMETER;

    err = sdc_key_len_get_bits_bytes(keylen_val, &bits, &bytes);

    if (err == SDC_OK) {
        *keylen_bytes = bytes;
    }

    return err;
}

sdc_error_t sdc_key_len_from_bits(size_t bits, sdc_key_len_t *keylen_val)
{
    sdc_error_t err;
    sdc_key_len_t iter;
    size_t len;

    if (!keylen_val)
        return SDC_INVALID_PARAMETER;

    err = sdc_key_len_bmsk_first(SDC_KEY_LEN_BMSK_ALL, &iter);
    while ((iter != SDC_KEY_LEN_UNKNOWN) && (err == SDC_OK)) {
        err = sdc_key_len_get_bits(iter, &len);
        if (err == SDC_OK) {
            if (bits == len) {
                *keylen_val = iter;
                break;
            }

            err = sdc_key_len_bmsk_next(SDC_KEY_LEN_BMSK_ALL, &iter);
        }
    }

    /* length not found */
    if ((err == SDC_OK) && (iter == SDC_KEY_LEN_UNKNOWN))
        err = SDC_KEYLEN_INVALID;

    return err;
}

sdc_error_t sdc_key_len_from_bytes(size_t bytes, sdc_key_len_t *keylen_val)
{
    sdc_error_t err;
    sdc_key_len_t iter;
    size_t len;

    if (!keylen_val)
        return SDC_INVALID_PARAMETER;

    err = sdc_key_len_bmsk_first(SDC_KEY_LEN_BMSK_ALL, &iter);
    while ((iter != SDC_KEY_LEN_UNKNOWN) && (err == SDC_OK)) {
        err = sdc_key_len_get_bytes(iter, &len);
        if (err == SDC_OK) {
            if (bytes == len) {
                *keylen_val = iter;
                break;
            }

            err = sdc_key_len_bmsk_next(SDC_KEY_LEN_BMSK_ALL, &iter);
        }
    }

    /* length not found */
    if ((err == SDC_OK) && (iter == SDC_KEY_LEN_UNKNOWN))
        err = SDC_KEYLEN_INVALID;

    return err;
}

sdc_error_t sdc_key_len_bmsk_next(sdc_key_len_bmsk_t keylen_bmsk, sdc_key_len_t *keylen_val)
{
    sdc_error_t err = SDC_OK;

    if (!keylen_val)
        return SDC_INVALID_PARAMETER;

    if (*keylen_val == SDC_KEY_LEN_UNKNOWN) {
        *keylen_val = SDC_KEY_LEN_FIRST;
    } else {
        err = check_sdc_key_len_t(*keylen_val);

        if (err == SDC_OK)
            *keylen_val = (sdc_key_len_t)(((uint32_t)*keylen_val) + 1);
    }

    if (err == SDC_OK) {
        while (((keylen_bmsk & SDC_KEY_LEN_TO_BMSK(*keylen_val)) == 0) &&
               (*keylen_val < SDC_KEY_LEN_END)) {
            *keylen_val = (sdc_key_len_t)(((uint32_t)*keylen_val) + 1);
        }

        if (*keylen_val >= SDC_KEY_LEN_END) {
            *keylen_val = SDC_KEY_LEN_UNKNOWN;
        }
    }

    return err;
}

sdc_error_t sdc_key_len_bmsk_first(sdc_key_len_bmsk_t keylen_bmsk, sdc_key_len_t *keylen_val)
{
    if (!keylen_val)
        return SDC_INVALID_PARAMETER;

    *keylen_val = SDC_KEY_LEN_UNKNOWN;

    return sdc_key_len_bmsk_next (keylen_bmsk, keylen_val);
}

static sdc_error_t check_sdc_key_fmt_t(sdc_key_fmt_t key_fmt)
{
    if ((key_fmt >= SDC_KEY_FMT_FIRST) && (key_fmt < SDC_KEY_FMT_END))
        return SDC_OK;

    return SDC_KEY_FMT_INVALID;
}

sdc_error_t sdc_key_fmt_bmsk_next(sdc_key_fmt_bmsk_t key_fmt_bmsk, sdc_key_fmt_t *key_fmt_val)
{
    sdc_error_t err = SDC_OK;

    if (!key_fmt_val)
        return SDC_INVALID_PARAMETER;

    if (*key_fmt_val == SDC_KEY_FMT_UNKNOWN) {
        *key_fmt_val = SDC_KEY_FMT_FIRST;
    } else {
        err = check_sdc_key_fmt_t(*key_fmt_val);

        if (err == SDC_OK)
            *key_fmt_val = (sdc_key_fmt_t)(((uint32_t)*key_fmt_val) + 1);
    }

    if (err == SDC_OK) {
        while (((key_fmt_bmsk & SDC_KEY_FMT_TO_BMSK(*key_fmt_val)) == 0) &&
               (*key_fmt_val < SDC_KEY_FMT_END)) {
            *key_fmt_val = (sdc_key_fmt_t)(((uint32_t)*key_fmt_val) + 1);
        }

        if (*key_fmt_val >= SDC_KEY_FMT_END) {
            *key_fmt_val = SDC_KEY_FMT_UNKNOWN;
        }
    }

    return err;
}

sdc_error_t sdc_key_fmt_bmsk_first(sdc_key_fmt_bmsk_t key_fmt_bmsk, sdc_key_fmt_t *key_fmt_val)
{
    if (!key_fmt_val)
        return SDC_INVALID_PARAMETER;

    *key_fmt_val = SDC_KEY_FMT_UNKNOWN;

    return sdc_key_fmt_bmsk_next (key_fmt_bmsk, key_fmt_val);
}

const char *sdc_key_fmt_get_name(sdc_key_fmt_t key_fmt)
{
    sdc_error_t err;
    size_t idx;
    size_t elems;

    err = check_sdc_key_fmt_t(key_fmt);
    if (err == SDC_OK) {
        idx = (size_t)key_fmt;
        elems = ARRAY_SIZE(sdc_key_fmt_names);

        if (idx<elems)
            return sdc_key_fmt_names[idx];
    }

    return NULL;
}

sdc_error_t sdc_key_len_bmsks(sdc_key_fmt_t key_fmt, sdc_key_len_bmsk_t *arch_keylen_bmsk)
{
    sdc_error_t err;

    if (!arch_keylen_bmsk)
        return SDC_INVALID_PARAMETER;

    err = check_sdc_key_fmt_t(key_fmt);
    if (err == SDC_OK)
        err = sdc_arch_key_len_bmsks(key_fmt, arch_keylen_bmsk);

    return err;
}


/* Common functions defined in sdc_intern.h (internal interface) */

